using System;
using System.Collections;
using System.Collections.Generic;
using Server;
using Server.Multis;
using Server.Regions;
using Server.Mobiles;

namespace Server.Items
{
	public enum AddonFitResult
	{
		Valid,
		Blocked,
		NotInHouse,
		DoorTooClose,
		NoWall,
		DoorsNotClosed
	}

	public interface IAddon
	{
		Item Deed{ get; }

		bool CouldFit( IPoint3D p, Map map );
	}

	public abstract class BaseAddon : Item, IChopable, IAddon
	{
		#region Mondain's Legacy
		private CraftResource m_Resource;

		[CommandProperty( AccessLevel.GameMaster )]
		public CraftResource Resource
		{
			get{ return m_Resource;	}
			set
			{
				if ( m_Resource != value )
				{
					m_Resource = value;
					Hue = CraftResources.GetHue( m_Resource );
					
					InvalidateProperties();
				}
			}
		}
		#endregion
		private List<AddonComponent> m_Components;

		public void AddComponent( AddonComponent c, int x, int y, int z )
		{
			if ( Deleted )
				return;

			m_Components.Add( c );

			c.Addon = this;
			c.Offset = new Point3D( x, y, z );
			c.MoveToWorld( new Point3D( X + x, Y + y, Z + z ), Map );
		}

		public BaseAddon() : base( 1 )
		{
			Movable = false;
			Visible = false;

			m_Components = new List<AddonComponent>();
		}

		public virtual bool RetainDeedHue{ get{ return false; } }

		public virtual void OnChop( Mobile from )
		{
			BaseHouse house = BaseHouse.FindHouseAt( this );
			
			PlayerMobile pm = (PlayerMobile)from; //NEW for Player city
			bool ismayor = false;
			if ( pm.City != null && pm.City.Mayor == pm && PlayerGovernmentSystem.IsAtCity( from ) )
				ismayor = true;

			if ( house != null && house.IsOwner( from ) && house.Addons.Contains( this ) || ismayor )
			{
				Effects.PlaySound( GetWorldLocation(), Map, 0x3B3 );
				from.SendLocalizedMessage( 500461 ); // You destroy the item.

				int hue = 0;

				if ( RetainDeedHue )
				{
					for ( int i = 0; hue == 0 && i < m_Components.Count; ++i )
					{
						AddonComponent c = m_Components[i];

						if ( c.Hue != 0 )
							hue = c.Hue;
					}
				}

				Delete();

				if ( house != null )
				house.Addons.Remove( this );

				BaseAddonDeed deed = Deed;

				if ( deed != null )
				{
					if ( RetainDeedHue )
						deed.Hue = hue;

					from.AddToBackpack( deed );
				}
			}
		}

		public virtual BaseAddonDeed Deed{ get{ return null; } }

		Item IAddon.Deed
		{
			get{ return this.Deed; }
		}

		public List<AddonComponent> Components
		{
			get
			{
				return m_Components;
			}
		}

		public BaseAddon( Serial serial ) : base( serial )
		{
		}

		public bool CouldFit( IPoint3D p, Map map )
		{
			List<BaseHouse> houses = null;

			return ( CouldFit( p, map, null, ref houses ) == AddonFitResult.Valid );
		}

		public AddonFitResult CouldFit( IPoint3D p, Map map, Mobile from, ref BaseHouse house ) //NEW Added for Compatibility
		{
			List<BaseHouse> houses = null;
			houses.Add( house );
			return CouldFit( p, map, from, ref houses );
		}
		
		public AddonFitResult CouldFit( IPoint3D p, Map map, Mobile from, ref List<BaseHouse> houses )
		{
			if ( PlayerGovernmentSystem.IsAtCity( this ) )
			    return AddonFitResult.Valid;
						
			if ( Deleted )
				return AddonFitResult.Blocked;
			
			if ( houses == null )
				houses = new List<BaseHouse>();

			foreach ( AddonComponent c in m_Components )
			{
				Point3D p3D = new Point3D( p.X + c.Offset.X, p.Y + c.Offset.Y, p.Z + c.Offset.Z );

				if ( !map.CanFit( p3D.X, p3D.Y, p3D.Z, c.ItemData.Height, false, true, ( c.Z == 0 ) ) )
					return AddonFitResult.Blocked;
				else if ( !CheckHouse( from, p3D, map, c.ItemData.Height, ref houses ) )
					return AddonFitResult.NotInHouse;

				if ( c.NeedsWall )
				{
					Point3D wall = c.WallPosition;

					if ( !IsWall( p3D.X + wall.X, p3D.Y + wall.Y, p3D.Z + wall.Z, map ) )
						return AddonFitResult.NoWall;
				}
			}

			foreach ( BaseHouse house in houses )
			{
				ArrayList doors = house.Doors;

				for ( int i = 0; i < doors.Count; ++i )
				{
					BaseDoor door = doors[i] as BaseDoor;

					
					Point3D doorLoc = door.GetWorldLocation();
					int doorHeight = door.ItemData.CalcHeight;

					foreach ( AddonComponent c in m_Components )
					{
						Point3D addonLoc = new Point3D( p.X + c.Offset.X, p.Y + c.Offset.Y, p.Z + c.Offset.Z );
						int addonHeight = c.ItemData.CalcHeight;

						if ( Utility.InRange( doorLoc, addonLoc, 1 ) && (addonLoc.Z == doorLoc.Z || ((addonLoc.Z + addonHeight) > doorLoc.Z && (doorLoc.Z + doorHeight) > addonLoc.Z)) )
							return AddonFitResult.DoorTooClose;
					}
				}
			}

			return AddonFitResult.Valid;
		}

		public static bool CheckHouse( Mobile from, Point3D p, Map map, int height, ref List<BaseHouse> list )
		{
			BaseHouse house = BaseHouse.FindHouseAt( p, map, height );
			
			if ( from == null )
				return true;
			if ( house == null )
				return false;
			if ( !house.IsOwner( from ) )
				return false;
			
			if ( !list.Contains( house ) )
				list.Add( house );
			
				return true;
					
		}
		
		public static bool CheckHouse( Mobile from, Point3D p, Map map, int height, ref BaseHouse house )
		{
			house = BaseHouse.FindHouseAt( p, map, height );

			if ( from == null || house == null || !house.IsOwner( from ) )
				return false;

			return true;
		}

		public static bool IsWall( int x, int y, int z, Map map )
		{
			if ( map == null )
				return false;

			StaticTile[] tiles = map.Tiles.GetStaticTiles( x, y, true );

			for ( int i = 0; i < tiles.Length; ++i )
			{
				StaticTile t = tiles[i];
				ItemData id = TileData.ItemTable[t.ID & TileData.MaxItemValue];

				if ( (id.Flags & TileFlag.Wall) != 0 && (z + 16) > t.Z && (t.Z + t.Height) > z )
					return true;
			}

			return false;
		}

		public virtual void OnComponentLoaded( AddonComponent c )
		{
		}

		public virtual void OnComponentUsed( AddonComponent c, Mobile from )
		{
		}

		public override void OnLocationChange( Point3D oldLoc )
		{
			if ( Deleted )
				return;

			foreach ( AddonComponent c in m_Components )
				c.Location = new Point3D( X + c.Offset.X, Y + c.Offset.Y, Z + c.Offset.Z );
		}

		public override void OnMapChange()
		{
			if ( Deleted )
				return;

			foreach ( AddonComponent c in m_Components )
				c.Map = Map;
		}

		public override void OnAfterDelete()
		{
			base.OnAfterDelete();

			foreach ( AddonComponent c in m_Components )
				c.Delete();
		}

		public virtual bool ShareHue{ get{ return true; } }

		[Hue, CommandProperty( AccessLevel.GameMaster )]
		public override int Hue
		{
			get
			{
				return base.Hue;
			}
			set
			{
				if ( base.Hue != value )
				{
					base.Hue = value;

					if ( !Deleted && this.ShareHue && m_Components != null )
					{
						foreach ( AddonComponent c in m_Components )
							c.Hue = value;
					}
				}
			}
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 1 ); // version

			writer.WriteItemList<AddonComponent>( m_Components );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
				case 0:
				{
					m_Components = reader.ReadStrongItemList<AddonComponent>();
					break;
				}
			}

			if ( version < 1 && Weight == 0 )
				Weight = -1;
		}
	}
}